import { Badge, Button, Col, Container, Form, Modal } from 'react-bootstrap';
import { MdCable } from 'react-icons/md';
import {
    AiOutlineDownload,
    AiOutlineUpload,
    AiOutlineDisconnect,
} from 'react-icons/ai';
import { FiEdit } from 'react-icons/fi';
import { VscDebugDisconnect } from 'react-icons/vsc';
import _ from 'lodash';
import { getColor } from '../../../../../../util/colorModule';
import { ConnectedComponent } from './ConnectedComponent';
import { useState } from 'react';
import toast from 'react-hot-toast';
import { useDispatch } from 'react-redux';
import { portUpdate } from '../../../../../../store/asyncMethods/deviceMethod';

function EditDevicePort({ portDetail, setDeviceDetail }) {
    const dispatch = useDispatch();
    const [updatedData, setUpdatedData] = useState({
        label: portDetail.label,
        port_type: portDetail.port_type,
        port_stream: portDetail.port_stream,
    });
    const [show, setShow] = useState(false);

    const handleClose = () => setShow(false);
    const handleShow = () => setShow(true);

    const onChangeHandler = (e) => {
        setUpdatedData((prevState) => ({
            ...prevState,
            [e.target.name]: e.target.value,
        }));
    };

    const onSubmitHandler = async (e) => {
        e.preventDefault();
        try {
            dispatch(
                portUpdate(portDetail.id, updatedData, (newData) => {
                    handleClose();
                    setDeviceDetail((prevState) => {
                        const ports = prevState.ports;
                        const portIndex = ports.findIndex(
                            (port) => port.id === portDetail.id
                        );
                        ports[portIndex] = {
                            ...ports[portIndex],
                            ...newData,
                        };
                        return {
                            ...prevState,
                            ports: [...ports],
                        };
                    });
                })
            );
            handleClose();
        } catch (error) {
            toast.error(error.message);
        }
    };

    return (
        <>
            <Badge
                className='bg-light text-dark text-light py-1'
                role='button'
                pill
                onClick={handleShow}
            >
                <FiEdit />
            </Badge>
            <Modal show={show} onHide={handleClose}>
                <Form onSubmit={onSubmitHandler}>
                    <Modal.Header closeButton>
                        <Modal.Title>Edit Port</Modal.Title>
                    </Modal.Header>
                    <Modal.Body>
                        <Form.Control
                            value={updatedData.label}
                            onChange={onChangeHandler}
                            name='label'
                        />
                        <Form.Select
                            value={updatedData.port_type}
                            onChange={onChangeHandler}
                            name='port_type'
                            className='mt-2'
                        >
                            <option value='1g'>1G</option>
                            <option value='10g'>10G</option>
                            <option value='40g'>40G</option>
                            <option value='100g'>100G</option>
                            <option value='pon'>PON</option>
                            <option value='xpon'>XPON</option>
                            <option value='other'>OTHER</option>
                        </Form.Select>

                        <Form.Select
                            value={updatedData.port_stream}
                            onChange={onChangeHandler}
                            className='mt-2'
                            name='port_stream'
                        >
                            <option value='incoming'>Incoming</option>
                            <option value='outgoing'>Outgoing</option>
                        </Form.Select>
                    </Modal.Body>
                    <Modal.Footer>
                        <Button variant='secondary' onClick={handleClose}>
                            Close
                        </Button>
                        <Button variant='primary' type='submit'>
                            Save Changes
                        </Button>
                    </Modal.Footer>
                </Form>
            </Modal>
        </>
    );
}

const DevicePortDetail = ({
    port,
    cluster,
    onPortDisconnect,
    setDeviceDetail,
}) => {
    return (
        <Col xl={6} xxl={4} lg={6} sm={12}>
            <Container fluid className='d-flex shadow-sm rounded pt-1 py-2'>
                <h5 className='bg-warning rounded me-1 p-2 d-flex align-items-center text-dark'>
                    {port.port_number}:{_.toUpper(port.port_type)}:
                    {port.port_stream === 'incoming' ? (
                        <AiOutlineDownload className='mb-1 mt-1' />
                    ) : (
                        <AiOutlineUpload className='mb-1 mt-1' />
                    )}
                </h5>{' '}
                <div>
                    <h6 className='m-0 '>
                        {port.label}{' '}
                        <EditDevicePort
                            portDetail={port}
                            setDeviceDetail={setDeviceDetail}
                        />
                    </h6>
                    {port.fiber ? (
                        <>
                            <Badge className='ms-1' bg='dark'>
                                <MdCable /> {port.fiber.cable.name}: (
                                {port.fiber.fiber_number}:
                                {_.toUpper(getColor(port.fiber.fiber_number))})
                            </Badge>
                            <Badge
                                pill
                                className='ms-1'
                                bg='danger'
                                role='button'
                                onClick={() => onPortDisconnect(port.id)}
                            >
                                <VscDebugDisconnect />
                            </Badge>
                        </>
                    ) : (
                        <>
                            <ConnectedComponent
                                connected_component={port.connected_component}
                                cluster={cluster}
                            />
                            {port.connected_component && (
                                <Badge
                                    pill
                                    className='ms-1'
                                    bg='danger'
                                    role='button'
                                    onClick={() => onPortDisconnect(port.id)}
                                >
                                    <AiOutlineDisconnect />
                                </Badge>
                            )}
                        </>
                    )}
                </div>
            </Container>
        </Col>
    );
};

export default DevicePortDetail;
