import { useState } from "react";
import { Col, Form, Row } from "react-bootstrap";
import { toast } from "react-hot-toast";
import _ from "lodash";
import axiosInstance from "../../../../../../util/axiosInstance";
import { getColor } from "../../../../../../util/colorModule";
import { Components } from "./Components";
import { ComponentOutput } from "./ComponentOutput";

const DeviceA = ({
    cluster,
    setCableFiber,
    setPort,
    deviceLocation,
    setDeviceLocation,
    componentOption,
    setComponentOption,
    component,
    setComponent,
    cable,
    setCable,
}) => {
    const [cableTube, setCableTube] = useState(null);

    const onDeviceLocationChangeHandler = async (event) => {
        if (isNaN(event.target.value)) {
            setDeviceLocation("local");
            setCableFiber(null);
            return;
        }
        setDeviceLocation(event.target.value);
        const { data } = await axiosInstance.get(
            `/cables/${event.target.value}`
        );
        setCable(data);
    };

    const onPortChangeHandler = (event) => {
        if (isNaN(event.target.value) && componentOption !== "onu") {
            setPort(null);
            return;
        } else if (componentOption === "onu") {
            setPort(event.target.value);
            return;
        }
        setPort(parseInt(event.target.value));
    };

    const onComponentChangeHandler = async (e) => {
        if (isNaN(e.target.value)) {
            setComponent(null);
            return;
        }
        setComponent(null);
        try {
            const componentType = {
                device: "devices",
                splitter: "splitters",
                onu: "onus",
            };
            const { data } = await axiosInstance.get(
                `/${componentType[componentOption]}/${e.target.value}`
            );
            setComponent(data);
        } catch (error) {
            toast.error(error.message);
        }
    };

    const onCableTubeChangeHandler = (e) => {
        if (isNaN(e.target.value)) {
            setCableTube(null);
            setCableFiber(null);
            return;
        }
        setCableFiber(null);
        const cableTube = _.find(cable.tubes, {
            id: parseInt(e.target.value),
        });
        setCableTube(cableTube);
    };

    const onCableFiberChangeHandler = (e) => {
        if (isNaN(e.target.value)) {
            setCableFiber(null);
            return;
        }

        setCableFiber(parseInt(e.target.value));
    };

    return (
        <>
            <h6>Component A</h6>
            <div className="w-75 mt-3 m-auto">
                <Form.Group>
                    <Form.Label>Device Location / Cable</Form.Label>
                    <Form.Select onChange={onDeviceLocationChangeHandler}>
                        <option value="local">Local</option>
                        {cluster.up_stream.map((cable) => (
                            <option value={cable.id} key={cable.id}>
                                {cable.name}
                            </option>
                        ))}
                        {cluster.down_stream.map((cable) => (
                            <option value={cable.id} key={cable.id}>
                                {cable.name}
                            </option>
                        ))}
                    </Form.Select>
                </Form.Group>

                {deviceLocation !== "local" ? (
                    <Row>
                        <Col>
                            <Form.Group className="mt-2">
                                <Form.Label>Select Cable Tube</Form.Label>
                                <Form.Select
                                    onChange={onCableTubeChangeHandler}
                                >
                                    <option>select tube</option>
                                    {cable?.tubes.map((tube) => (
                                        <option value={tube.id} key={tube.id}>
                                            {tube.tube_number} (
                                            {getColor(tube.tube_number)})
                                        </option>
                                    ))}
                                </Form.Select>
                            </Form.Group>
                        </Col>
                        <Col>
                            {cableTube && (
                                <Form.Group className="mt-2">
                                    <Form.Label>Select Cable Fiber</Form.Label>
                                    <Form.Select
                                        onChange={onCableFiberChangeHandler}
                                    >
                                        <option>select fiber</option>
                                        {cableTube.fibers.map((fiber) => (
                                            <option
                                                value={fiber.id}
                                                key={fiber.id}
                                            >
                                                {fiber.fiber_number} (
                                                {getColor(fiber.fiber_number)}){" "}
                                                {fiber.child_fiber_id ||
                                                fiber.child_component_id
                                                    ? " - Used"
                                                    : " - Available"}
                                            </option>
                                        ))}
                                    </Form.Select>
                                </Form.Group>
                            )}
                        </Col>
                    </Row>
                ) : (
                    <>
                        <Form.Group className="mt-2">
                            <Form.Label>Select Component Type</Form.Label>
                            <Form.Select
                                onChange={(e) => {
                                    setComponentOption(e.target.value);
                                    setPort(null);
                                    setComponent(null);
                                }}
                            >
                                <option>Select Component</option>
                                <option value="device">Device</option>
                                <option value="splitter">Splitter</option>
                                <option value="onu">ONU</option>
                            </Form.Select>
                        </Form.Group>
                        <Components
                            devices={cluster.devices}
                            splitters={cluster.splitters}
                            onus={cluster.onus}
                            onComponentChangeHandler={onComponentChangeHandler}
                            componentOption={componentOption}
                        />
                        <ComponentOutput
                            onPortChangeHandler={onPortChangeHandler}
                            componentOption={componentOption}
                            component={component}
                        />
                    </>
                )}
            </div>
        </>
    );
};

export default DeviceA;
